program CUBICSPLINE;
{--------------------------------------------------------------------}
{  Alg5'4.pas   Pascal program for implementing Algorithm 5.4        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 5.4 (Cubic Splines).                                    }
{  Section   5.3, Interpolation by Spline Functions, Page 297        }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxN = 50;
    GNmax = 630;

  type
    VECTOR = array[0..MaxN] of real;
    RVECTOR = array[0..GNmax] of real;
    MATRIX = array[0..MaxN, 0..3] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, More, SawTable, Working);
    DATYPE = (DatPoints, FunPoints);
    ABTYPE = (Given, Equal, Interval, Chebyshev);
  var
    DNpts, GNpts, Inum, N, Stype, Sub: integer;
    DX0, DXN, DDX0, DDXN, Rnum, T: real;
    Xmax, Xmin, Ymax, Ymin: real;
    A, B, C, D, H, M, V, X, Y: VECTOR;
    Xg, Yg: RVECTOR;
    S: MATRIX;
    Ans, Resp: CHAR;
    Mess: LETTERS;
    Stat, State, Statt: STATUS;
    Ytype: DATYPE;
    Xtype: ABTYPE;

  procedure DIFFERENCES (X, Y: VECTOR; N: integer; var A, B, C, D, H, V: VECTOR);
    var
      K: integer;
  begin
    H[0] := X[1] - X[0];                          {Difference in abscissas}
    D[0] := (Y[1] - Y[0]) / H[0];                        {Difference quotient}
    for K := 1 to N - 1 do
      begin
        H[K] := X[K + 1] - X[K];                   {Differences in abscissas}
        D[K] := (Y[K + 1] - Y[K]) / H[K];                  {Difference quotient}
        A[K] := H[K];                               {Subdiagonal elements}
        B[K] := 2 * (H[K - 1] + H[K]);                       {Diagonal elements}
        C[K] := H[K];                             {Superdiagonal elements}
        V[K] := 6 * (D[K] - D[K - 1]);                       {The column vector}
      end;
  end;

  procedure MODIFY (var A, B, C: VECTOR; D, H: VECTOR; var M, V: VECTOR; N, Stype: integer; var DX0, DXN, DDX0, DDXN: real);
  begin
    case Stype of
      1: 
        begin
          CLRSCR;
          WRITELN;
          WRITELN;
          WRITELN('    You chose to construct the clamped spline.');
          WRITELN;
          WRITELN;
          WRITELN('Now you must specify S`(X ) and S`(X ).');
          WRITELN('                         0          N');
          WRITELN;
          WRITELN('ENTER the first derivatives:');
          WRITELN;
          WRITELN;
          WRITELN;
          WRITE('S`(', X[0] : 15 : 7, '  ) = ');
          DX0 := 0;
          READLN(DX0);
          WRITELN;
          WRITE('S`(', X[N] : 15 : 7, '  ) = ');
          DXN := 0;
          READLN(DXN);
          WRITELN;
          B[1] := B[1] - H[0] / 2;
          V[1] := V[1] - 3 * (D[0] - DX0);
          B[N - 1] := B[N - 1] - H[N - 1] / 2;
          V[N - 1] := V[N - 1] - 3 * (DXN - D[N - 1]);
        end;
      2: 
        begin
          M[0] := 0;
          M[N] := 0;
        end;
      3: 
        begin
          B[1] := B[1] + H[0] + H[0] * H[0] / H[1];
          C[1] := C[1] - H[0] * H[0] / H[1];
          B[N - 1] := B[N - 1] + H[N - 1] + H[N - 1] * H[N - 1] / H[N - 2];
          A[N - 2] := A[N - 2] - H[N - 1] * H[N - 1] / H[N - 2];
        end;
      4: 
        begin
          B[1] := B[1] + H[0];
          B[N - 1] := B[N - 1] + H[N - 1];
        end;
      5: 
        begin
          CLRSCR;
          WRITELN;
          WRITELN;
          WRITELN('You chose to construct the spline which will have');
          WRITELN;
          WRITELN;
          WRITELN('specified values of  S``(X )  and  S``(X ) .');
          WRITELN('                      0             N ');
          WRITELN;
          WRITELN('ENTER the second derivatives:');
          WRITELN;
          WRITELN;
          WRITELN;
          WRITE('S``(', X[0] : 15 : 7, '  ) = ');
          DDX0 := 0;
          READLN(DDX0);
          WRITELN;
          WRITE('S``(', X[N] : 15 : 7, '  ) = ');
          DDXN := 0;
          READLN(DDXN);
          WRITELN;
          V[1] := V[1] - H[0] * DDX0;
          V[N - 1] := V[N - 1] - H[N - 1] * DDXN;
        end;
    end;
  end;

  procedure LINEARSYS (A, B, C, V: VECTOR; var M: VECTOR; N: integer);
    var
      K: integer;
      T: real;
    label
      999;
  begin                                  {Gaussian elimination is used}
    if N < 2 then
      goto 999;
    for K := 2 to N - 1 do
      begin                               {to get the upper-triangular}
        T := A[K - 1] / B[K - 1];
        B[K] := B[K] - T * C[K - 1];             {system with "two diagonals."}
        V[K] := V[K] - T * V[K - 1];
      end;
    M[N - 1] := V[N - 1] / B[N - 1];                       {Back substitution is}
    for K := N - 2 downto 1 do
      M[K] := (V[K] - C[K] * M[K + 1]) / B[K];           {used to solve for M(K)}
999:
  end;

  procedure ENDCOEFFICIENT (D, H: VECTOR; var M: VECTOR; N, Stype: integer; DX0, DXN, DDX0, DDXN: real);
  begin
    case Stype of
      1: 
        begin
          M[0] := 3 * (D[0] - DX0) / H[0] - M[1] / 2;
          M[N] := 3 * (DXN - D[N - 1]) / H[N - 1] - M[N - 1] / 2;
        end;
      2: 
        begin
          M[0] := 0;
          M[N] := 0;
        end;
      3: 
        begin
          M[0] := M[1] - H[0] * (M[2] - M[1]) / H[1];
          M[N] := M[N - 1] + H[N - 1] * (M[N - 1] - M[N - 2]) / H[N - 2];
        end;
      4: 
        begin
          M[0] := M[1];
          M[N] := M[N - 1];
        end;
      5: 
        begin
          M[0] := DDX0;
          M[N] := DDXN;
        end;
    end;
  end;

  procedure COMPUTECOEFF (D, H, M: VECTOR; var S: MATRIX; N: integer);
    var
      I, K: integer;
  begin
    for K := 0 to N - 1 do
      begin
        S[K, 0] := Y[K];
        S[K, 1] := D[K] - H[K] * (2 * M[K] + M[K + 1]) / 6;
        S[K, 2] := M[K] / 2;
        S[K, 3] := (M[K + 1] - M[K]) / (6 * H[K]);
      end;
  end;

  function CS (S: MATRIX; X: VECTOR; N: integer; T: real): real;
    var
      J, K: integer;
      W: real;
  begin
    K := N - 1;
    for J := 1 to N do
      if (X[J - 1] <= T) and (T <= X[J]) then
        begin
          K := J - 1;
        end;
    if T <= X[0] then
      K := 0;
    W := T - X[K];
    CS := ((S[K, 3] * W + S[K, 2]) * W + S[K, 1]) * W + S[K, 0];
  end;

  procedure GETPOINTS (var X, Y: VECTOR; var Xmin, Xmax: real; var N: integer; Stat: STATUS);
    type
      STATUS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      Count, I, J, K, Kbad: integer;
      T, Valu: real;
      Resp: CHAR;
      Cond: STATUS;
  begin
    CLRSCR;
    Kbad := -1;
    State := Working;
    if Stat = More then
      begin
        for I := 1 to 6 do
          WRITELN;
        WRITE('Do  you  want  to  enter   new   data   points ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Working;
            CLRSCR;
          end;
      end;
    if (Stat = Working) then
      begin
        CLRSCR;
        Kbad := 0;
        for K := 0 to N do
          begin
            X[K] := 0;
            Y[K] := 0;
          end;
        CLRSCR;
        WRITELN;
        Kbad := -1;
        WRITELN;
        WRITELN('          Now enter the  ', N + 1 : 2, '  points.');
        WRITELN;
        WRITELN('          You will have a chance to make changes at the end.');
        WRITELN;
        WRITELN;
        for K := 0 to N do
          begin
            X[K] := 0;
            Y[K] := 0;
          end;
        Xtype := Given;
        for K := 0 to N do
          begin
            if Xtype = Given then
              begin
                WRITELN;
                Mess := '         x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
              end;
            Ytype := DatPoints;
            if Ytype = DatPoints then
              begin
                if Xtype <> Given then
                  begin
                    WRITELN;
                    WRITELN('         x  =', X[K]);
                    WRITE('          ', K : 0);
                  end;
                Mess := '         y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
                WRITELN;
              end
            else
              begin
                     {Y[K]:=F(X[K]); Provision for function values.}
              end;
            WRITELN;
          end;
      end;
    Xmin := X[0];
    Ymin := Y[0];
    for K := 0 to N do
      begin
        if (Xmin > X[K]) then
          Xmin := X[K];
        if (Ymin > Y[K]) then
          Ymin := Y[K];
      end;
    Cond := Enter;
    while (Cond = Enter) or (Cond = Bad) do
      begin
        CLRSCR;
        if (Cond = Bad) then
          WRITELN('     The abscissas are NOT distinct.   You MUST change one of them.');
        WRITELN('      k               x                     y');
        WRITELN('                       k                     k');
        WRITELN('----------------------------------------------------------------');
        for K := 0 to N do
          WRITELN('     ', K : 2, '       ', X[K] : 15 : 7, '       ', Y[K] : 15 : 7);
        WRITELN;
        if (Cond <> Bad) then
          begin
            WRITELN;
            if N > 15 then
              begin
                for I := 1 to 9 do
                  WRITELN;
              end;
            WRITE('     Are the points o.k. ?  <Y/N>  ');
            READLN(Resp);
            WRITELN;
          end;
        if (Resp = 'N') or (Resp = 'n') or (Cond = Bad) then
          begin
            if N > 14 then
              begin
                WRITELN;
              end;
            WRITELN;
            WRITELN;
            case N of
              1: 
                WRITELN('     To change a point select  k = 0,1');
              2: 
                WRITELN('     To change a point select  k = 0,1,2');
              else
                WRITELN('     To change a point select  k = 0,1,...,', N : 2);
            end;
            Mess := '                       ENTER   k = ';
            K := Kbad;
            WRITE(Mess);
            READLN(K);
            if (0 <= K) and (K <= N) then
              begin
                WRITELN;
                if K < 10 then
                  begin
                    WRITELN('     Coordinates of the  current point  (x ,y )  are:');
                    WRITELN('                                          ', K : 1, '  ', k : 1);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '      Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 1, '                              ', K : 1);
                  end
                else
                  begin
                    WRITELN('     Coordinates of the current point  (x  ,y  )  are:');
                    WRITELN('                                         ', K : 2, '  ', k : 2);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '      Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 2, '                             ', K : 2);
                  end;
                Mess := '     NEW   x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
                Mess := '     NEW   y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
              end;
          end
        else
          Cond := Done;
        if (Cond = Bad) then
          Cond := Enter;
        Kbad := -1;
        Count := 0;
        for J := 0 to N - 1 do
          for K := J + 1 to N do
            if (X[J] = X[K]) then
              begin
                Kbad := K;
                Cond := Bad;
              end;
        for J := 0 to N - 1 do
          begin
            for K := J + 1 to N do
              if X[J] > X[K] then
                begin
                  T := X[J];
                  X[J] := X[K];
                  X[K] := T;
                  T := Y[J];
                  Y[J] := Y[K];
                  Y[K] := T;
                end;
          end;
        Xmax := X[0];
        Xmin := X[0];
        Ymax := Y[0];
        Ymin := Y[0];
        for K := 1 to N do
          begin
            if (Xmax < X[K]) then
              Xmax := X[K];
            if (Xmin > X[K]) then
              Xmin := X[K];
            if (Ymax < Y[K]) then
              Ymax := Y[K];
            if (Ymin > Y[K]) then
              Ymin := Y[K];
          end;
      end;
  end;

  procedure INPUTS (var X, Y: VECTOR; var N: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     A cubic spline is constructed which passes through');
    WRITELN;
    WRITELN;
    WRITELN('     the  N+1  points  (X ,Y ) , (X ,Y ) ,..., (X ,Y ).');
    WRITELN('                         0  0      1  1          N  N ');
    WRITELN;
    WRITELN('     The solution is a set of  N  piecewise cubic functions:');
    WRITELN;
    WRITELN('                        3             2');
    WRITELN('     S (X) =  S   (X-X )  + S   (X-X )  + S   (X-X ) + S    ,');
    WRITELN('      K        K,3    K      K,2    K      K,1    K     K,0  ');
    WRITELN;
    WRITELN('     where  X  is in  [X ,X   ]  and  K = 0,1,...,N-1.');
    WRITELN('                        K  K+1 ');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('     There are N+1 points.');
    WRITELN;
    Mess := '     ENTER the number  N = ';
    N := 1;
    WRITE(Mess);
    READLN(N);
    if N < 1 then
      N := 1;
    if N > 50 then
      N := 50;
    WRITELN;
    GETPOINTS(X, Y, Xmin, Xmax, N, Stat);
  end;

  procedure SPLINETYPE (var Stype: integer; N: integer);
    var
      Ans: CHAR;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('     There are five types of splines that can be constructed:');
    WRITELN;
    WRITELN;
    WRITELN('     1 - Clamped spline, specify  S`(X )  and  S`(X ) .');
    WRITELN('                                      0            N');
    WRITELN;
    WRITELN('     2 - Natural spline, i.e. S``(X ) = 0 and S``(X ) = 0 .');
    WRITELN('                                   0               N');
    WRITELN;
    WRITELN('     3 - Extrapolate S``(X) near the endpoints.');
    WRITELN;
    WRITELN;
    WRITELN('     4 - Parabolic near the endpoints.');
    WRITELN;
    WRITELN;
    WRITELN('     5 - Specify  S``(X )  and  S``(X ) .');
    WRITELN('                       0             N');
    WRITELN;
    WRITELN;
    Mess := '     Select type < 1 - 5 >  ';
    Stype := 1;
    WRITE(Mess);
    READLN(Stype);
    if Stype < 1 then
      Stype := 1;
    if Stype > 5 then
      Stype := 5;
    if (Stype = 3) and (N = 1) then
      begin
        CLRSCR;
        WRITELN;
        WRITELN('     There are not enough points to, hence S``(X)');
        WRITELN;
        WRITELN('     cannot be extrapolated near the endpoints.');
        WRITELN;
        WRITELN('     Therefore I will choose the natural cubic spline.');
        WRITELN;
        Stype := 2;
        WRITELN('     Press the <ENTER> key. ');
        READLN(Ans);
        WRITELN;
      end;
  end;

  procedure RESULTS (S: MATRIX; X, Y: VECTOR; N, Stype: integer; DX0, DXN, DDX0, DDXN: real);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    case Stype of
      1: 
        begin
          WRITELN('You chose to construct the clamped spline with S`(X ) and S`(X ) specified.');
          WRITELN('                                                   0          N');
        end;
      2: 
        WRITELN('You chose to construct the natural spline.');
      3: 
        WRITELN('You chose to construct the spline with S``(X) extrapolated near the endpoints.');
      4: 
        WRITELN('You chose to construct the spline which is parabolic near the endpoints.');
      5: 
        begin
          WRITELN('You chose to construct the spline with S``(X ) and S``(X ) specified.');
          WRITELN('                                        0           N');
        end;
    end;
    if (Stype <> 1) and (Stype <> 5) then
      begin
        WRITELN(' K        X              Y ');
        WRITELN('           K              K');
      end;
    if Stype = 1 then
      begin
        WRITELN(' K        X              Y              S`(X )');
        WRITELN('           K              K                 K');
      end;
    if Stype = 5 then
      begin
        WRITELN(' K        X              Y              S``(X )');
        WRITELN('           K              K                  K');
      end;
    for K := 0 to N do
      begin
        WRITE(K : 2, X[K] : 15 : 7, Y[K] : 15 : 7);
        if (Stype = 1) and (K = 0) then
          WRITELN(DX0 : 15 : 7);
        if (Stype = 1) and (K = N) then
          WRITELN(DXN : 15 : 7);
        if (Stype = 5) and (K = 0) then
          WRITELN(DDX0 : 15 : 7);
        if (Stype = 5) and (K = N) then
          WRITELN(DDXN : 15 : 7);
        if (Stype <> 1) and (Stype <> 5) then
          WRITELN
        else if (0 < K) and (K < N) then
          WRITELN;
      end;
    WRITELN;
    WRITELN('                                     3              2');
    WRITELN('The spline is: S (X)  =  S   (X - X ) + S   (X - X ) + S   (X - X ) + S   :');
    WRITELN('                K         K,3      K     K,2      K     K,1      K     K,0');
    WRITELN(' K        X              S              S              S              S');
    WRITELN('           K              K,3            K,2            K,1            K,0');
    for K := 0 to N - 1 do
      WRITELN(K : 2, X[K] : 15 : 7, S[K, 3] : 15 : 7, S[K, 2] : 15 : 7, S[K, 1] : 15 : 7, S[K, 0] : 15 : 7);
    WRITELN;
  end;

  procedure EVALUATE (S: MATRIX; X, Y: VECTOR; N, Stype: integer; DX0, DXN, DDX0, DDXN: real);
    var
      T: real;
      Ans: CHAR;
  begin
    Ans := 'y';
    CLRSCR;
    while (Ans = 'y') or (Ans = 'Y') do
      begin
        WRITELN;
        WRITE('Do  you  want to  evaluate  the   spline  S(x) ?  <Y/N>  ');
        READLN(Ans);
        WRITELN;
        if (Ans = 'y') or (Ans = 'Y') then
          begin
            WRITELN;
            Mess := '     ENTER a value  x = ';
            T := 0;
            WRITE(Mess);
            READLN(T);
            WRITELN;
            WRITELN('The spline`s value is  S( ', T : 0 : 5, ' )  =  ', CS(S, X, N, T) : 0 : 5);
          end;
      end;
  end;

  procedure MESSAGE;
    var
      Ans: CHAR;
  begin
    CLRSCR;
    WRITELN('                             CUBIC SPLINES');
    WRITELN;
    WRITELN;
    WRITELN('   Consider N+1 points {(x ,y )} where  x  < x  < ... < x . The function S(x)');
    WRITELN('                          k  k           0    1          N  ');
    WRITELN('is called a cubic spline if there exists N cubic polynomials S (x) such that:');
    WRITELN('                                                              k');
    WRITELN('                                            2            3   ');
    WRITELN('S(x) = S (x) = s   + s   (x-x ) + s   (x-x ) + s   (x-x )   for x in [x ,x   ]');
    WRITELN('        k       k,0   k,1    k     k,2    k     k,3    k               k  k+1 ');
    WRITELN('                                                            and k = 0,1,...,N-1');
    WRITELN;
    WRITELN('       S(x ) = y             for  k = 0,1,...,N');
    WRITELN('          k     k');
    WRITELN;
    WRITELN('    S (x   ) = S   (x   )    for  k = 0,1,...,N-2');
    WRITELN('     k  k+1     k+1  k+1');
    WRITELN;
    WRITELN('    S`(x   ) = S`  (x   )    for  k = 0,1,...,N-2');
    WRITELN('     k  k+1     k+1  k+1');
    WRITELN;
    WRITELN('  S`` (x   ) = S`` (x   )    for  k = 0,1,...,N-2');
    WRITELN('   k    k+1     k+1  k+1');
    WRITELN;
    WRITE('                             Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure TABLE (S: MATRIX; X, Y: VECTOR; N, Stype: integer; DX0, DXN, DDX0, DDXN: real; var State: STATUS);
    var
      J, ML, MN: integer;
      A0, B0, H, T: real;
  begin
    CLRSCR;
    WRITELN;
    if State = Computing then
      begin
        WRITE('Want to construct  a table of values for  S(X) ?  <Y/N>  ');
        READLN(Ans);
        WRITELN;
      end;
    if State = More then
      Ans := 'Y';
    if (Ans = 'Y') or (Ans = 'y') then
      begin
        State := SawTable;
        CLRSCR;
        WRITELN;
        WRITELN('A table of M+1 equally spaced point will be made in [a,b].');
        Mess := '    ENTER the  left endpoint  a = ';
        A0 := Xmin;
        WRITE(Mess);
        READLN(A0);
        Mess := '    ENTER the right endpoint  b = ';
        B0 := Xmax;
        WRITE(Mess);
        READLN(B0);
        Mess := 'ENTER number of subintervals  M = ';
        MN := 10;
        WRITE(Mess);
        READLN(MN);
        if MN < 1 then
          MN := 1;
        if MN > 100 then
          MN := 100;
        WRITELN;
        H := (B0 - A0) / MN;
        CLRSCR;
        WRITELN;
        WRITELN('                 x                     S(x )');
        WRITELN('                  k                       k');
        WRITELN('     -------------------------------------------');
        ML := 0;
        for J := ML to MN do
          begin
            T := A0 + H * J;
            WRITELN('     ', T : 17 : 5, '     ', CS(S, X, N, T) : 17 : 5);
          end;
      end;
  end;

begin                                            {Begin Main Program}
  Stat := Working;
  MESSAGE;
  while (Stat = Working) do
    begin
      INPUTS(X, Y, N);
      State := Computing;
      while (State = Computing) do
        begin
          DIFFERENCES(X, Y, N, A, B, C, D, H, V);
          SPLINETYPE(Stype, N);
          MODIFY(A, B, C, D, H, M, V, N, Stype, DX0, DXN, DDX0, DDXN);
          LINEARSYS(A, B, C, V, M, N);
          ENDCOEFFICIENT(D, H, M, N, Stype, DX0, DXN, DDX0, DDXN);
          COMPUTECOEFF(D, H, M, S, N);
          RESULTS(S, X, Y, N, Stype, DX0, DXN, DDX0, DDXN);
          WRITE('Press the <ENTER> key. ');
          READLN(Resp);
          WRITELN;
          EVALUATE(S, X, Y, N, Stype, DX0, DXN, DDX0, DDXN);
          WRITELN;
          WRITE('Press the <ENTER> key. ');
          READLN(Resp);
          WRITELN;
          Statt := Computing;
          while (Statt = Computing) or (Statt = More) or (Statt = Sawtable) do
            begin
              CLRSCR;
              TABLE(S, X, Y, N, Stype, DX0, DXN, DDX0, DDXN, State);
              if Statt = SawTable then
                begin
                  WRITELN;
                  WRITE('Do you  want  to see another  table  of values ?  <Y/N>  ');
                  READLN(Ans);
                  WRITELN;
                end
              else
                begin
                  Ans := 'N';
                  WRITELN;
                end;
              if (Ans <> 'Y') and (Ans <> 'y') then
                Statt := Done;
              if (Ans = 'Y') or (Ans = 'y') then
                Statt := More;
            end;
          WRITELN;
          WRITELN;
          WRITE('Want to find a different spline for this  data ?  <Y/N>  ');
          READLN(Resp);
          WRITELN;
          if (Resp <> 'Y') and (Resp <> 'y') then
            State := Done;
          if (Resp = 'Y') or (Resp = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want to run the program again using new points ?  <Y/N>  ');
      READLN(Resp);
      WRITELN;
      if (Resp <> 'Y') and (Resp <> 'y') then
        Stat := Done
    end;
end.                                           {End Main Program}

